<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Berita;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BeritaController extends Controller
{
    public function index()
    {
        $beritaList = Berita::orderBy('created_at', 'desc')->paginate(10);
        
        return view('admin.berita.index', compact('beritaList'));
    }

    public function create()
    {
        return view('admin.berita.form');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'judul' => 'required|string|max:255',
            'ringkasan' => 'nullable|string|max:500',
            'isi' => 'required|string',
            'gambar' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'tanggal_publikasi' => 'required|date',
            'status' => 'required|in:draft,published',
        ]);

        $validated['author_id'] = Auth::id();
        $validated['slug'] = Str::slug($validated['judul']);

        if ($request->hasFile('gambar')) {
            $file = $request->file('gambar');
            $filename = time() . '_' . Str::slug($validated['judul']) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('images/berita'), $filename);
            $validated['gambar'] = 'berita/' . $filename;
        }

        Berita::create($validated);

        return redirect()->route('admin.berita.index')->with('success', 'Berita berhasil ditambahkan!');
    }

    public function edit($id)
    {
        $berita = Berita::findOrFail($id);
        
        return view('admin.berita.form', compact('berita'));
    }

    public function update(Request $request, $id)
    {
        $berita = Berita::findOrFail($id);

        $validated = $request->validate([
            'judul' => 'required|string|max:255',
            'ringkasan' => 'nullable|string|max:500',
            'isi' => 'required|string',
            'gambar' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'tanggal_publikasi' => 'required|date',
            'status' => 'required|in:draft,published',
        ]);

        if ($request->hasFile('gambar')) {
            // Delete old image if exists
            if ($berita->gambar && file_exists(public_path('images/' . $berita->gambar))) {
                unlink(public_path('images/' . $berita->gambar));
            }
            
            $file = $request->file('gambar');
            $filename = time() . '_' . Str::slug($validated['judul']) . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('images/berita'), $filename);
            $validated['gambar'] = 'berita/' . $filename;
        }

        $berita->update($validated);

        return redirect()->route('admin.berita.index')->with('success', 'Berita berhasil diperbarui!');
    }

    public function destroy($id)
    {
        $berita = Berita::findOrFail($id);

        // Delete image if exists
        if ($berita->gambar && file_exists(public_path('images/' . $berita->gambar))) {
            unlink(public_path('images/' . $berita->gambar));
        }

        $berita->delete();

        return redirect()->route('admin.berita.index')->with('success', 'Berita berhasil dihapus!');
    }

    public function toggleStatus($id)
    {
        $berita = Berita::findOrFail($id);
        $berita->status = $berita->status === 'published' ? 'draft' : 'published';
        $berita->save();

        $statusText = $berita->status === 'published' ? 'dipublikasikan' : 'disimpan sebagai draft';
        
        return redirect()->route('admin.berita.index')->with('success', "Berita berhasil {$statusText}!");
    }
}
