<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Jemaat;
use App\Models\FormulirDoa;
use App\Models\FormulirKonseling;
use App\Models\Keuangan;
use App\Models\JadwalIbadah;
use App\Models\Donasi;
use App\Models\PesanKontak;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        $jemaatOnline = Jemaat::aktif()->count();
        $permohonanDoa = FormulirDoa::where('status', 'baru')->count();
        
        $bulanIni = Carbon::now()->format('Y-m');
        $donasiBulanIni = Keuangan::pemasukan()->periode($bulanIni)->sum('jumlah');
        
        $konselingPending = FormulirKonseling::where('status', 'pending')->count();

        $notifikasi = collect();
        
        $doaBaru = FormulirDoa::where('status', 'baru')
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'doa',
                    'pesan' => 'Permohonan doa baru dari ' . $item->nama,
                    'waktu' => $item->created_at->diffForHumans(),
                ];
            });
        
        $konselingBaru = FormulirKonseling::where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->take(2)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'konseling',
                    'pesan' => 'Permohonan konseling dari ' . $item->nama,
                    'waktu' => $item->created_at->diffForHumans(),
                ];
            });
        
        $donasiBaru = Donasi::where('status', 'dikonfirmasi')
            ->orderBy('created_at', 'desc')
            ->take(2)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'donasi',
                    'pesan' => 'Konfirmasi donasi Rp ' . number_format($item->jumlah, 0, ',', '.') . ' dari ' . $item->nama,
                    'waktu' => $item->created_at->diffForHumans(),
                ];
            });

        $pesanKontakBaru = PesanKontak::belumDibaca()
            ->orderBy('created_at', 'desc')
            ->take(3)
            ->get()
            ->map(function ($item) {
                return [
                    'type' => 'pesan',
                    'pesan' => 'Pesan baru dari ' . $item->nama . ': ' . $item->subjek,
                    'waktu' => $item->created_at->diffForHumans(),
                ];
            });
        
        $notifikasi = $doaBaru->concat($konselingBaru)->concat($donasiBaru)->concat($pesanKontakBaru)
            ->sortByDesc('waktu')
            ->take(5)
            ->values()
            ->toArray();

        if (empty($notifikasi)) {
            $notifikasi = [
                ['type' => 'info', 'pesan' => 'Belum ada notifikasi baru', 'waktu' => 'Baru saja'],
            ];
        }

        $jadwalMingguIni = JadwalIbadah::where('aktif', true)
            ->orderBy('hari')
            ->orderBy('waktu')
            ->take(5)
            ->get()
            ->map(function ($item) {
                $hariMap = ['Minggu' => 0, 'Senin' => 1, 'Selasa' => 2, 'Rabu' => 3, 'Kamis' => 4, 'Jumat' => 5, 'Sabtu' => 6];
                $hariIndex = $hariMap[$item->hari] ?? 0;
                $tanggal = Carbon::now()->startOfWeek()->addDays($hariIndex);
                
                return [
                    'nama' => $item->nama,
                    'tanggal' => $item->hari . ', ' . $tanggal->format('d M'),
                    'waktu' => substr($item->waktu, 0, 5),
                    'pelayan' => $item->pelayan ?? '-',
                    'online' => true,
                ];
            })->toArray();

        return view('admin.dashboard', compact('jemaatOnline', 'permohonanDoa', 'donasiBulanIni', 'konselingPending', 'notifikasi', 'jadwalMingguIni'));
    }
}
