<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProfilGereja;
use App\Models\Pendeta;
use App\Models\Majelis;
use App\Models\RekeningBank;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class ProfilController extends Controller
{
    public function index()
    {
        $profilModel = ProfilGereja::getProfil();
        $profil = [
            'nama_gereja' => $profilModel->nama_gereja ?? 'Gereja Jemaat Buntu Ambaang',
            'alamat' => $profilModel->alamat ?? '',
            'telepon' => $profilModel->telepon ?? '',
            'email' => $profilModel->email ?? '',
            'tahun_berdiri' => $profilModel->tahun_berdiri ?? '',
            'visi' => $profilModel->visi ?? '',
            'misi' => $profilModel->misi ?? [],
        ];

        $pendetaModel = Pendeta::getPendeta();
        $pendeta = [
            'nama' => $pendetaModel->nama ?? '',
            'jabatan' => $pendetaModel->jabatan ?? 'Pendeta Jemaat',
            'telepon' => $pendetaModel->telepon ?? '',
            'email' => $pendetaModel->email ?? '',
            'mulai_melayani' => $pendetaModel->mulai_melayani ?? '',
            'foto' => $pendetaModel->foto_path,
        ];

        $majelisData = Majelis::ordered()->get();
        $majelis = $majelisData->map(function ($item) {
            return [
                'id' => $item->id,
                'nama' => $item->nama,
                'jabatan' => $item->jabatan,
                'bidang' => $item->bidang,
                'foto' => $item->foto_path,
            ];
        })->toArray();

        $rekeningList = RekeningBank::ordered()->get()->map(function ($item) {
            return [
                'id' => $item->id,
                'nama_bank' => $item->nama_bank,
                'nomor_rekening' => $item->nomor_rekening,
                'atas_nama' => $item->atas_nama,
                'kode_bank' => $item->kode_bank,
                'is_primary' => $item->is_primary,
                'aktif' => $item->aktif,
            ];
        })->toArray();

        return view('admin.profil.index', compact('profil', 'pendeta', 'majelis', 'rekeningList'));
    }

    public function updateProfil(Request $request)
    {
        $request->validate([
            'nama_gereja' => 'required|string|max:255',
            'alamat' => 'required|string',
        ]);

        $profil = ProfilGereja::first();
        if (!$profil) {
            $profil = new ProfilGereja();
        }

        $profil->fill([
            'nama_gereja' => $request->nama_gereja,
            'alamat' => $request->alamat,
            'telepon' => $request->telepon,
            'email' => $request->email,
            'tahun_berdiri' => $request->tahun_berdiri,
            'visi' => $request->visi,
            'misi' => $request->misi ? array_filter(explode("\n", $request->misi)) : [],
        ]);
        $profil->save();

        return redirect()->route('admin.profil.index')->with('success', 'Profil gereja berhasil diperbarui');
    }

    public function updatePendeta(Request $request)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);

        $pendeta = Pendeta::first();
        if (!$pendeta) {
            $pendeta = new Pendeta();
        }

        $pendeta->fill([
            'nama' => $request->nama,
            'jabatan' => $request->jabatan ?? 'Pendeta Jemaat',
            'telepon' => $request->telepon,
            'email' => $request->email,
            'mulai_melayani' => $request->mulai_melayani,
        ]);

        if ($request->hasFile('foto')) {
            $this->deleteOldPhoto('pendeta');
            $file = $request->file('foto');
            $filename = 'pendeta.' . $file->getClientOriginalExtension();
            $file->move(public_path('images/profil'), $filename);
            $pendeta->foto = 'profil/' . $filename;
        }

        $pendeta->save();
        
        return redirect()->route('admin.profil.index')->with('success', 'Data pendeta berhasil diperbarui');
    }

    public function deletePendetaPhoto()
    {
        $pendeta = Pendeta::first();
        if ($pendeta) {
            $this->deleteOldPhoto('pendeta');
            $pendeta->foto = null;
            $pendeta->save();
        }
        return redirect()->route('admin.profil.index')->with('success', 'Foto pendeta berhasil dihapus');
    }

    public function storeMajelis(Request $request)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'jabatan' => 'required|string|max:255',
            'bidang' => 'required|string|max:255',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);

        $maxUrutan = Majelis::max('urutan') ?? 0;

        $majelis = Majelis::create([
            'nama' => $request->nama,
            'jabatan' => $request->jabatan,
            'bidang' => $request->bidang,
            'urutan' => $maxUrutan + 1,
        ]);

        if ($request->hasFile('foto')) {
            $file = $request->file('foto');
            $filename = "majelis-{$majelis->id}." . $file->getClientOriginalExtension();
            $file->move(public_path('images/profil'), $filename);
            $majelis->foto = 'profil/' . $filename;
            $majelis->save();
        }
        
        return redirect()->route('admin.profil.index')->with('success', 'Anggota majelis berhasil ditambahkan');
    }

    public function updateMajelis(Request $request, $id)
    {
        $request->validate([
            'nama' => 'required|string|max:255',
            'jabatan' => 'required|string|max:255',
            'bidang' => 'required|string|max:255',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);

        $majelis = Majelis::findOrFail($id);
        $majelis->fill([
            'nama' => $request->nama,
            'jabatan' => $request->jabatan,
            'bidang' => $request->bidang,
        ]);

        if ($request->hasFile('foto')) {
            $this->deleteOldPhoto("majelis-{$id}");
            $file = $request->file('foto');
            $filename = "majelis-{$id}." . $file->getClientOriginalExtension();
            $file->move(public_path('images/profil'), $filename);
            $majelis->foto = 'profil/' . $filename;
        }

        $majelis->save();
        
        return redirect()->route('admin.profil.index')->with('success', 'Data majelis berhasil diperbarui');
    }

    public function deleteMajelisPhoto($id)
    {
        $majelis = Majelis::findOrFail($id);
        $this->deleteOldPhoto("majelis-{$id}");
        $majelis->foto = null;
        $majelis->save();

        return redirect()->route('admin.profil.index')->with('success', 'Foto majelis berhasil dihapus');
    }

    public function destroyMajelis($id)
    {
        $majelis = Majelis::findOrFail($id);
        $this->deleteOldPhoto("majelis-{$id}");
        $majelis->delete();

        return redirect()->route('admin.profil.index')->with('success', 'Anggota majelis berhasil dihapus');
    }

    private function deleteOldPhoto($identifier)
    {
        $extensions = ['jpg', 'jpeg', 'png', 'webp'];
        foreach ($extensions as $ext) {
            $path = public_path("images/profil/{$identifier}.{$ext}");
            if (File::exists($path)) {
                File::delete($path);
            }
        }
    }

    public function storeRekening(Request $request)
    {
        $request->validate([
            'nama_bank' => 'required|string|max:255',
            'nomor_rekening' => 'required|string|max:50',
            'atas_nama' => 'required|string|max:255',
        ]);

        $maxUrutan = RekeningBank::max('urutan') ?? 0;

        RekeningBank::create([
            'nama_bank' => $request->nama_bank,
            'nomor_rekening' => $request->nomor_rekening,
            'atas_nama' => $request->atas_nama,
            'kode_bank' => $request->kode_bank,
            'is_primary' => $request->has('is_primary'),
            'aktif' => $request->has('aktif') || !$request->has('_method'),
            'urutan' => $maxUrutan + 1,
        ]);

        return redirect()->route('admin.profil.index')->with('success', 'Rekening bank berhasil ditambahkan');
    }

    public function updateRekening(Request $request, $id)
    {
        $request->validate([
            'nama_bank' => 'required|string|max:255',
            'nomor_rekening' => 'required|string|max:50',
            'atas_nama' => 'required|string|max:255',
        ]);

        $rekening = RekeningBank::findOrFail($id);
        $rekening->update([
            'nama_bank' => $request->nama_bank,
            'nomor_rekening' => $request->nomor_rekening,
            'atas_nama' => $request->atas_nama,
            'kode_bank' => $request->kode_bank,
            'is_primary' => $request->has('is_primary'),
            'aktif' => $request->has('aktif'),
        ]);

        return redirect()->route('admin.profil.index')->with('success', 'Rekening bank berhasil diperbarui');
    }

    public function destroyRekening($id)
    {
        RekeningBank::findOrFail($id)->delete();
        return redirect()->route('admin.profil.index')->with('success', 'Rekening bank berhasil dihapus');
    }
}
