<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Jemaat;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function index()
    {
        $users = User::orderBy('name')->paginate(10);

        $totalJemaat = Jemaat::aktif()->count();
        $totalPending = Jemaat::pending()->count();
        
        $jemaatAktif = Jemaat::aktif()->orderBy('nama')->paginate(10, ['*'], 'jemaat_page');
        $jemaatPending = Jemaat::pending()->terbaru()->paginate(10, ['*'], 'pending_page');
        $jemaatDitolak = Jemaat::ditolak()->terbaru()->paginate(10, ['*'], 'ditolak_page');

        return view('admin.user.index', compact('users', 'totalJemaat', 'totalPending', 'jemaatAktif', 'jemaatPending', 'jemaatDitolak'));
    }

    public function create()
    {
        return view('admin.user.form');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:6',
            'role' => 'required|string',
        ]);

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('admin.user.index')->with('success', 'Pengguna berhasil ditambahkan.');
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);
        return view('admin.user.form', compact('user'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'role' => 'required|string',
        ]);

        $user = User::findOrFail($id);
        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
            'is_active' => $request->has('is_active'),
        ];
        
        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);

        return redirect()->route('admin.user.index')->with('success', 'Pengguna berhasil diupdate.');
    }

    public function destroy($id)
    {
        User::findOrFail($id)->delete();
        return redirect()->route('admin.user.index')->with('success', 'Pengguna berhasil dihapus.');
    }

    public function reset($id)
    {
        $user = User::findOrFail($id);
        $user->update(['password' => Hash::make('password123')]);
        return redirect()->route('admin.user.index')->with('success', 'Password berhasil direset ke: password123');
    }

    public function storeJemaat(Request $request)
    {
        $request->validate(
            Jemaat::validationRules(),
            Jemaat::validationMessages()
        );

        Jemaat::create([
            'kode' => Jemaat::generateKode(),
            'nama' => $request->nama,
            'jenis_kelamin' => $request->jenis_kelamin,
            'tempat_lahir' => $request->tempat_lahir,
            'tanggal_lahir' => $request->tanggal_lahir,
            'alamat' => $request->alamat,
            'telepon' => $request->telepon,
            'email' => $request->email,
            'status_pernikahan' => $request->status_pernikahan,
            'pekerjaan' => $request->pekerjaan,
            'status' => 'aktif',
            'aktif' => true,
        ]);

        return redirect()->route('admin.user.index')->with('success', 'Jemaat berhasil ditambahkan.');
    }

    public function updateJemaat(Request $request, $id)
    {
        $request->validate(
            Jemaat::validationRules(),
            Jemaat::validationMessages()
        );

        $jemaat = Jemaat::findOrFail($id);
        $jemaat->update([
            'nama' => $request->nama,
            'jenis_kelamin' => $request->jenis_kelamin,
            'tempat_lahir' => $request->tempat_lahir,
            'tanggal_lahir' => $request->tanggal_lahir,
            'alamat' => $request->alamat,
            'telepon' => $request->telepon,
            'email' => $request->email,
            'status_pernikahan' => $request->status_pernikahan,
            'pekerjaan' => $request->pekerjaan,
        ]);

        return redirect()->route('admin.user.index')->with('success', 'Data jemaat berhasil diupdate.');
    }

    public function updateStatusJemaat(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,aktif,ditolak',
        ]);

        $jemaat = Jemaat::findOrFail($id);
        $jemaat->update([
            'status' => $request->status,
            'aktif' => $request->status === 'aktif',
        ]);

        $statusText = [
            'pending' => 'Pending',
            'aktif' => 'Aktif',
            'ditolak' => 'Ditolak'
        ];

        return redirect()->route('admin.user.index')->with('success', "Status jemaat berhasil diubah menjadi {$statusText[$request->status]}.");
    }

    public function showJemaat($id)
    {
        $jemaat = Jemaat::findOrFail($id);
        return response()->json($jemaat);
    }

    public function destroyJemaat($id)
    {
        Jemaat::findOrFail($id)->delete();
        return redirect()->route('admin.user.index')->with('success', 'Jemaat berhasil dihapus.');
    }
}
