<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Berita extends Model
{
    protected $table = 'berita';

    protected $fillable = [
        'judul',
        'slug',
        'ringkasan',
        'isi',
        'gambar',
        'tanggal_publikasi',
        'status',
        'aktif',
        'author_id',
    ];

    protected $casts = [
        'tanggal_publikasi' => 'date',
        'aktif' => 'boolean',
    ];

    public function getRouteKeyName()
    {
        return 'slug';
    }

    public function author()
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    public function scopePublished($query)
    {
        return $query->where('status', 'published')
                     ->where('aktif', true)
                     ->where('tanggal_publikasi', '<=', now());
    }

    public function scopeTerbaru($query, $limit = 10)
    {
        return $query->published()->orderBy('tanggal_publikasi', 'desc')->limit($limit);
    }

    public function scopeAktif($query)
    {
        return $query->where('aktif', true);
    }

    public static function boot()
    {
        parent::boot();

        static::creating(function ($berita) {
            if (empty($berita->slug)) {
                $berita->slug = Str::slug($berita->judul);
            }
            $originalSlug = $berita->slug;
            $count = 1;
            while (static::where('slug', $berita->slug)->exists()) {
                $berita->slug = $originalSlug . '-' . $count++;
            }
        });

        static::updating(function ($berita) {
            if ($berita->isDirty('judul') && !$berita->isDirty('slug')) {
                $berita->slug = Str::slug($berita->judul);
                $originalSlug = $berita->slug;
                $count = 1;
                while (static::where('slug', $berita->slug)->where('id', '!=', $berita->id)->exists()) {
                    $berita->slug = $originalSlug . '-' . $count++;
                }
            }
        });
    }

    public function getRingkasanAttribute($value)
    {
        if ($value) {
            return $value;
        }
        return Str::limit(strip_tags($this->attributes['isi'] ?? ''), 150);
    }

    public function getIsiPlainAttribute()
    {
        return strip_tags($this->isi);
    }
}
