<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Jemaat extends Model
{
    protected $table = 'jemaat';

    protected $fillable = [
        'kode',
        'nama',
        'jenis_kelamin',
        'tempat_lahir',
        'tanggal_lahir',
        'alamat',
        'telepon',
        'email',
        'status_pernikahan',
        'pekerjaan',
        'aktif',
        'status',
    ];

    protected $casts = [
        'tanggal_lahir' => 'date',
        'aktif' => 'boolean',
    ];

    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeDitolak($query)
    {
        return $query->where('status', 'ditolak');
    }

    public function scopeTerbaru($query)
    {
        return $query->orderBy('created_at', 'desc');
    }

    public static function generateKode()
    {
        $tahun = date('Y');
        $bulan = date('m');
        $prefix = "JMT-{$tahun}{$bulan}-";
        
        $lastJemaat = self::where('kode', 'like', $prefix . '%')
            ->orderBy('kode', 'desc')
            ->first();
        
        if ($lastJemaat) {
            $lastNumber = (int) substr($lastJemaat->kode, -4);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }
        
        return $prefix . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    public static function jumlahPending()
    {
        return self::pending()->count();
    }

    public static function validationRules($isPublic = true)
    {
        return [
            'nama' => 'required|string|max:255',
            'jenis_kelamin' => 'required|in:L,P',
            'tempat_lahir' => 'required|string|max:255',
            'tanggal_lahir' => 'required|date|before:today',
            'alamat' => 'required|string|max:1000',
            'telepon' => 'required|string|max:20|regex:/^[0-9+\-\s]+$/',
            'email' => 'nullable|email|max:255',
            'status_pernikahan' => 'required|in:Belum Menikah,Menikah,Duda,Janda',
            'pekerjaan' => 'required|string|max:255',
        ];
    }

    public static function validationMessages()
    {
        return [
            'nama.required' => 'Nama lengkap wajib diisi',
            'jenis_kelamin.required' => 'Jenis kelamin wajib dipilih',
            'jenis_kelamin.in' => 'Jenis kelamin tidak valid',
            'tempat_lahir.required' => 'Tempat lahir wajib diisi',
            'tanggal_lahir.required' => 'Tanggal lahir wajib diisi',
            'tanggal_lahir.before' => 'Tanggal lahir harus sebelum hari ini',
            'alamat.required' => 'Alamat wajib diisi',
            'telepon.required' => 'Nomor telepon wajib diisi',
            'telepon.regex' => 'Format nomor telepon tidak valid',
            'email.email' => 'Format email tidak valid',
            'status_pernikahan.required' => 'Status pernikahan wajib dipilih',
            'status_pernikahan.in' => 'Status pernikahan tidak valid',
            'pekerjaan.required' => 'Pekerjaan wajib diisi',
        ];
    }
}
